<?php

namespace Drupal\ai_agents\Service;

use Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutputInterface;
use Drupal\ai_agents\Artifact\Artifact;
use Drupal\ai_agents\Artifact\ArtifactStorageInterface;
use Drupal\Component\Datetime\TimeInterface;

/**
 * Helper service for managing artifacts in AI agent operations.
 *
 * This service provides methods to store, retrieve, and replace artifacts
 * generated by tools during AI agent interactions.
 */
class ArtifactHelper {

  public function __construct(
    protected readonly ArtifactStorageInterface $storage,
    protected readonly TimeInterface $time,
  ) {}

  /**
   * Stores an artifact with the given tool ID and value.
   *
   * @param string $tool_id
   *   The ID of the tool that generated the artifact.
   * @param mixed $value
   *   The value of the artifact, which can be any type.
   *
   * @return \Drupal\ai_agents\Artifact\Artifact
   *   The stored artifact object.
   */
  public function store(string $tool_id, mixed $value): Artifact {
    $index = $this->storage->getNextIndex($tool_id);
    $artifact = new Artifact($tool_id, $index, $value, $this->time->getRequestTime());
    $this->storage->store($tool_id, $index, $artifact);
    return $artifact;
  }

  /**
   * Retrieves an artifact based on the placeholder string.
   *
   * The placeholder should be in the format `!artifact:tool_id:index`.
   *
   * @param string $placeholder
   *   The placeholder string to parse.
   *
   * @return \Drupal\ai_agents\Artifact\Artifact|null
   *   The retrieved artifact, or NULL if not found.
   */
  public function get(string $placeholder): ?Artifact {
    // Match placeholders like {{artifact:tool_id:1}}.
    if (preg_match('/^{{artifact:([^:}]+):(\d+)}}$/', $placeholder, $m)) {
      return $this->storage->get($m[1], (int) $m[2]);
    }
    return NULL;
  }

  /**
   * Replaces artifact placeholders in tool arguments with artifact values.
   *
   * This method scans the tool's arguments for any artifact placeholders and
   * replaces them with the corresponding artifact values.
   *
   * @param \Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutputInterface $tool
   *   The tool output containing arguments to process.
   */
  public function replaceArtifactArguments(ToolsFunctionOutputInterface $tool) {
    // Replace any artifact placeholders actual values.
    $arguments = $tool->getArguments();
    foreach ($arguments as $toolsPropertyResult) {
      $value = $toolsPropertyResult->getValue();
      // If the value is a string and matches the artifact placeholder format,
      // attempt to retrieve the artifact.
      if (is_string($value) && ($artifact = $this->get($value))) {
        $toolsPropertyResult->setValue($artifact->getValue());
      }
    }
  }

}
